
/* Copyright (C) 2001-2009 Monotype Imaging Inc. All rights reserved. */

/* Confidential Information of Monotype Imaging Inc. */

/* fs_effects.c */

#include "fs_itype.h"
#include "fs_effects.h"

/*************************************************************************/
/* Bitmap Special Effects functions                                      */
/*************************************************************************/

#if defined(FS_BITMAPS)

/*************************************************************************/
/*lint -e661 Warning -- Possible access of out-of-bounds pointer         */
/*lint -e662 Warning -- Possible creation of out-of-bounds pointer       */
/*************************************************************************/
/* engrave effect - light appears from lower right corner                */
/* a bit that is 1 stays 1 if the bit to the NW is 0, else it's set to 0 */
FS_BITMAP *engrave_bitmap(FS_BITMAP *bmap)
{
    FS_BYTE *bits, *prev;
    FS_SHORT height = bmap->height;
    FS_SHORT bpl = bmap->bpl;
    FS_SHORT i, j;
    FS_SHORT shl = 7;
    FS_SHORT shr = 1;

    /* start at the bottom right FS_BYTE */
    bits = bmap->bits + height * bpl - 1;
    prev = bits - bpl;

    for (i = height; i > 1; i--)
    {
        for (j = bpl; j > 1; j--)
        {
            *bits &= ~ ((*prev >> shr) | (FS_BYTE)(prev[-1] << shl));
            bits--;
            prev--;
        }
        *bits &= ~ (*prev >> shr);
        bits--;
        prev--;
    }
    return bmap;
}
/*************************************************************************/
/* emboss effect - light appears from upper right corner                 */
/* a bit that is 1 stays 1 if the bit to the SE is 0, else it's set to 0 */
FS_BITMAP *emboss_bitmap(FS_BITMAP *bmap)
{
    FS_BYTE *bits, *next;
    FS_SHORT height = bmap->height - 1;
    FS_SHORT bpl = bmap->bpl;
    FS_SHORT i, j;
    FS_SHORT shr = 7;
    FS_SHORT shl = 1;

    bits = bmap->bits;
    next = bits + bpl;

    bpl--;

    for (i = 0; i < height; i++)
    {
        for (j = 0; j < bpl; j++)
        {
            *bits &= ~ ((FS_BYTE)(*next << shl) | (next[1] >> shr));
            bits++;
            next++;
        }
        *bits &= ~ ((FS_BYTE)(*next << shl));
        bits++;
        next++;
    }
    return bmap;
}

/*************************************************************************/
/* add <nRows> to top and bottom and <nCols> to left and right of <src>  */
static FS_BITMAP *grow_bitmap(_DS_ FS_BITMAP *src, int nRows, int nCols)
{
    int width = src->width + nCols * 2;
    int height = src->height + nRows * 2;
    int bpl = (width + 7) / 8;
    int base, size, lsh, rsh, r, c;
    FS_BITMAP *dst;
    FS_BYTE *sp, *dp;

    base = sizeof(FS_BITMAP) - 1;
    size = height * bpl;
    dst = (FS_BITMAP *) FSS_calloc(_PS_ base + size);
    if (dst == 0)
        return dst;

    /* set header info */
    SYS_MEMCPY(dst, src, base - SIZEOF_LONG); /* see fs_port.h */
    dst->size = base + size;
    dst->cache_ptr = 0;
    dst->embedded = 0;
    dst->width = (FS_SHORT)width;
    dst->height = (FS_SHORT)height;
    dst->bpl = (FS_SHORT)bpl;
    dst->lo_x -= (FS_SHORT)nCols;
    dst->hi_y += (FS_SHORT)nRows;

    /* copy the bits */
    sp = src->bits;
    dp = dst->bits + nRows * bpl + (nCols >> 3);
    rsh = nCols & 7;
    lsh = 8 - rsh;
    height = src->height;
    bpl = src->bpl;
    for (r = 0; r < height; r++)
    {
        /* the first ... because there is no sp[-1] */
        dp[0] = sp[0] >> rsh;

        /* the middles */
        for (c = 1; c < bpl; c++)
            dp[c] = (sp[c - 1] << lsh) | (sp[c] >> rsh);

        /* the last ... because there is no sp[bpl] */
        dp[c] = sp[c - 1] << lsh;

        dp += dst->bpl;
        sp += bpl;
    }

    return dst;
}

/**********************************************************************************/
/* add <n> pixels to the 'thickness' of the black elements of <src>               */
/* alternating 8-neighbor and 4-neighbor gives better shape than 4-neighbor alone */
static FS_BITMAP *dilate_bitmap(_DS_ FS_BITMAP *src, int n)
{
    FS_BITMAP *dst;
    FS_BYTE *temp_bits;
    int r, h, c, w, bpl;

    /* added rows and columns eliminate "off the edge" testing. */
    dst = grow_bitmap(_PS_ src, n + 1, n + 8);
    if (dst == 0)
        return 0;

    bpl = dst->bpl;
    h = dst->height;
    temp_bits = FSS_calloc(_PS_ bpl * h);
    if (0 == temp_bits)
        return 0;

    /* don't use the last column or last row */
    w = bpl - 1;
    h--;

    while (n)
    {
        FS_BYTE *cr, *nr, *pr, *tp;

        tp = temp_bits + bpl;  /* test pixel   */
        pr = dst->bits;        /* previous row */
        cr = pr + bpl;         /* current row  */
        nr = cr + bpl;         /* next row      */

        if ( !(n & 1) )
        {
            /* fetch 8 neighbors */
            for (r = 1; r < h; r++)
            {
                for (c = 1; c < w; c++)
                {
                    tp[c]  = (cr[c]   << 1) | (cr[c + 1] >> 7); /* east */
                    tp[c] |= (cr[c - 1] << 7) | (cr[c]   >> 1); /* west */
                    tp[c] |= (pr[c]   << 1) | (pr[c + 1] >> 7); /* north-east */
                    tp[c] |= (pr[c - 1] << 7) | (pr[c]   >> 1); /* north-west */
                    tp[c] |= (nr[c]   << 1) | (nr[c + 1] >> 7); /* south-east */
                    tp[c] |= (nr[c - 1] << 7) | (nr[c]   >> 1); /* south-west */
                    tp[c] |= pr[c] | nr[c];                   /* north + south */
                }
                cr += bpl;
                pr += bpl;
                nr += bpl;
                tp += bpl;
            }
        }
        else
        {
            /* fetch 4 neighbors */
            for (r = 1; r < h; r++)
            {
                for (c = 1; c < w; c++)
                {
                    tp[c]  = (cr[c]   << 1) | (cr[c + 1] >> 7); /* east */
                    tp[c] |= (cr[c - 1] << 7) | (cr[c]   >> 1); /* west */
                    tp[c] |= pr[c] | nr[c];                   /* north + south */
                }
                cr += bpl;
                pr += bpl;
                nr += bpl;
                tp += bpl;
            }
        }

        /* do the dilate ... in new block to aid the compiler */
        /* we should be able to get SOME register variables   */
        {
            register int size = dst->bpl * dst->height;
            register FS_BYTE *dp = dst->bits;
            register FS_BYTE *mp = temp_bits;
            while (size--)
            {
                *dp |= (~*dp) & *mp++;
                dp++;
            }
        }
        n--;
    }
    FSS_free(_PS_ temp_bits);
    return dst;
}

/****************************************************************/
/* these bit macros expect <ptr> to be on the correct row       */
/* and they do not do bounds checking, so they are LOTS         */
/* faster than the old functions get_bit() and set_bit()        */
/*                                                              */
/* watch out for side effects on <c> -- it is evaluated twice.  */
/****************************************************************/
#define GB(ptr,c) (ptr[(c) >> 3] &   fs_mask[(c) & 7])
/*#define SB(ptr,c) (ptr[(c) >> 3] |=  fs_mask[(c) & 7])*/ /* not used*/
#define CB(ptr,c) (ptr[(c) >> 3] &= ~fs_mask[(c) & 7])

/****************************************************************/
/* clear bits in <a> which are set in <b>                       */
/* pay attention to <lo_x,hy_y> offset of each                  */
static void remove_bitmap(FS_BITMAP *a, FS_BITMAP *b)
{
    FS_BYTE *a_ptr, *b_ptr;
    int a_height, b_height, height;
    int a_width, b_width, width;
    int a_off, b_off;
    int r, c, dx, dy;

    a_ptr = a->bits;
    b_ptr = b->bits;
    a_height = a->height;
    b_height = b->height;
    a_width = a->width;
    b_width = b->width;

    dy = a->hi_y - b->hi_y;
    if (dy > 0)
    {
        a_height -= dy;
        a_ptr += dy * a->bpl;
    }
    else
    {
        b_height += dy;
        b_ptr -= dy * b->bpl;
    }

    dx = a->lo_x - b->lo_x;
    if (dx < 0)
    {
        a_width += dx;
        a_off = -dx;
        b_off = 0;
    }
    else
    {
        b_width -= dx;
        a_off = 0;
        b_off = dx;
    }

    height = MIN(a_height, b_height);
    width = MIN(a_width, b_width);
    for (r = 0; r < height; r++)
    {
        for (c = 0; c < width; c++)
        {
            if (GB(b_ptr, c + b_off))
                CB(a_ptr, c + a_off);
        }
        a_ptr += a->bpl;
        b_ptr += b->bpl;
    }
}
/*lint +e661 Warning -- Possible access of out-of-bounds pointer   */
/*lint +e662 Warning -- Possible creation of out-of-bounds pointer */

/***************************************************************************/
FS_BITMAP *outline_bitmap(_DS_ FS_BITMAP *src, FS_USHORT n, FS_USHORT filled)
{
    FS_BITMAP *dst = 0;
    if (src->width || src->height)
    {
        dst = dilate_bitmap(_PS_ src, n);
        if (dst && !filled)
            remove_bitmap(dst, src);
        FSS_free_char(_PS_ src);
        return dst;
    }
    else
    {
        src->lo_x = n;
        src->hi_y = -n;
        return src;
    }
}

#ifdef FS_PSEUDO_BOLD

/****************************************************************/
static FS_BITMAP *copy_bitmap(_DS_ FS_BITMAP *p)
{
    FS_BITMAP *r;

    if (p == 0)
        return 0;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_BITMAP";
#endif
    r = (FS_BITMAP *)FSS_malloc(_PS_ p->size );
    if (r == 0)
        return 0;

    SYS_MEMCPY(r, p, p->size);

    /* this is no longer a cached object */
    r->cache_ptr = 0;

    return r;
}

/***************************************************************************/
FS_BITMAP *pixelbold_bitmap(_DS_ FS_BITMAP *bmap)
{
    FS_BYTE *ptr;
    FS_BYTE left, me, right, prior;
    FS_USHORT c;
    FS_USHORT height;
    FS_USHORT bpl;
    FS_USHORT last;
    FS_LONG bold_width;

    bold_width = STATE.cur_sfnt->senv->bold_width;

    if (bold_width == 0)
        return bmap;

    if (bmap == 0)
        return bmap;

    height = bmap->height;
    bpl = bmap->bpl;
    last = bpl - 1;

    while (height--)
    {
        ptr = bmap->bits + (bmap->height - height - 1) * bpl;

        /* possibly add pixels on the left side */
        /* do the 0th byte */
        me = ptr[0];
        left = ptr[0] >> 1;              /* pixel to the left aligns with me      */

        if (last >= 1 )
            right = (ptr[0] << 1) | (ptr[1] >> 7);
        else
            right = (ptr[0] << 1);

        ptr[0] |= (right & ~(me | left)); /* set me if right=1 and me=0 and left=0 */

        /* do the middle bytes */
        for (c = 1; c < last; c++)
        {
            me = ptr[c];
            left = (ptr[c - 1] << 7) | (ptr[c] >> 1); /* pixel to the left aligns with me      */
            right = (ptr[c] << 1) | (ptr[c + 1] >> 7); /* pixel to the right aligns with me     */
            ptr[c] |= (right & ~(me | left));  /* set me if right=1 and me=0 and left=0 */
        }

        /* do the last byte */
        if (last >= 1)
        {
            me = ptr[last];
            left = (ptr[last - 1] << 7) | (ptr[last] >> 1); /* pixel to the left aligns with me      */
            right = ptr[last] << 1;                 /* pixel to the right aligns with me     */
            ptr[last] |= (right & ~(me | left));    /* set me if right=1 and me=0 and left=0 */
        }
    }

    if (bold_width >= 2 )
    {
        /* possibly add pixels on the right side */
        height = bmap->height;
        while (height--)
        {
            ptr = bmap->bits + (bmap->height - height - 1) * bpl;

            /* do the 0th byte */
            me = prior = ptr[0];
            left = ptr[0] >> 1;
            /* pixel to the left aligns with me      */

            if (last >= 1 )
                right = (ptr[0] << 1) | (ptr[1] >> 7);
            else
                right = (ptr[0] << 1);

            ptr[0] |= (left & ~(me | right)); /* set me if left=1 and me=0 and right=0 */

            /* do the middle bytes */
            for (c = 1; c < last; c++)
            {
                me = ptr[c];
                left = (prior << 7) | (ptr[c] >> 1); /* pixel to the left aligns with me      */
                right = (ptr[c] << 1) | (ptr[c + 1] >> 7); /* pixel to the right aligns with me     */
                prior = ptr[c];                    /* save byte before change               */
                ptr[c] |= (left & ~(me | right));  /* set me if left=1 and me=0 and right=0 */
            }

            /* do the last byte */
            if (last >= 1)
            {
                me = ptr[last];
                left = (prior << 7) | (ptr[last] >> 1); /* pixel to the left aligns with me      */
                right = ptr[last] << 1;             /* pixel to the right aligns with me     */
                ptr[last] |= (left & ~(me | right)); /* set me if left=1 and me=0 and right=0 */
            }
        }
    }

    return bmap;
}

/****************************************************************/
FS_BITMAP *pixelbold_embedded_bitmap(_DS_ FS_BITMAP *bmap)
{
    SFNT *sfnt = STATE.cur_sfnt;
    FS_SHORT bw = sfnt->senv->bold_width;
    FS_BITMAP *new_bmap, *cp_new;
    FS_SHORT width, height, bpl, row, col;
    FS_LONG size;
    FS_BYTE *src_bits, *dst_bits;
    int bit_added = 0;

    if (bw == 0)
        return bmap;

    if (bw == 1)
        bit_added = 1;
    else
        bit_added = 2;

    width = bmap->width + (FS_SHORT)bit_added;
    height = bmap->height;
    bpl = (7 + width) / 8;
    size = offsetof(FS_BITMAP, bits);
    size += height * bpl;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_BITMAP";
#endif
    new_bmap = (FS_BITMAP *) FSS_calloc(_PS_ size);
    if (new_bmap == 0)
    {
        FSS_free_char(_PS_ bmap);
        return 0;
    }
    new_bmap->size = size;
    new_bmap->width = width;
    new_bmap->height = height;
    new_bmap->bpl = bpl;
    new_bmap->i_dx = bmap->i_dx;
    new_bmap->i_dy = bmap->i_dy;
    new_bmap->dx = bmap->dx;
    new_bmap->dy = bmap->dy;
    new_bmap->hi_y = bmap->hi_y;
    new_bmap->lo_x = bmap->lo_x - 1;
    new_bmap->type = bmap->type;
    src_bits = bmap->bits;
    dst_bits = new_bmap->bits;
    for (row = 0; row < bmap->height; row++)
    {
        SYS_MEMCPY(dst_bits, src_bits, bmap->bpl);
        src_bits += bmap->bpl;
        dst_bits += new_bmap->bpl;
    }

    cp_new = copy_bitmap(_PS_ new_bmap);
    if (cp_new == 0)
    {
        FSS_free_char(_PS_ bmap);
        return 0;
    }
    for (row = 0; row < new_bmap->height; row++)
    {
        src_bits = cp_new->bits + row * bpl;
        dst_bits = new_bmap->bits + row * bpl;

        dst_bits[0] = src_bits[0] >> 1;
        for (col = 1; col < bpl; col++)
            dst_bits[col] = (src_bits[col - 1] << 7) | (src_bits[col] >> 1);
    }
    FSS_free_char(_PS_ bmap);
    FSS_free_char(_PS_ cp_new);
    pixelbold_bitmap(_PS_ new_bmap);
    return new_bmap;
}
#endif /* FS_PSEUDO_BOLD */

#endif /* FS_BITMAPS */

/*************************************************************************/
/* Graymap Special Effects functions                                     */
/* These also apply to embedded graymaps and Edge graymaps               */
/*************************************************************************/

#if defined(FS_GRAYMAPS) || defined(FS_EMBEDDED_BITMAP) || defined(FS_EDGE_RENDER)

/****************************************************************/
/* engrave effect - light appears from lower right corner       */
/* subtract NW pixel and clip negative values                   */
FS_GRAYMAP *engrave_graymap(FS_GRAYMAP *gmap)
{
    FS_BYTE *bits, *prev;
    FS_SHORT bpl;
    FS_SHORT h;

    if (gmap == 0)
        return 0;

    if ((gmap->height < 2) || (gmap->width < 2))
        return gmap;

    bpl = gmap->bpl;
    h = gmap->height - 1;

    if (gmap->bitsPerPixel == 8)
    {
        int j = h;
        /* start at the last row and work up */
        bits = gmap->bits + (h * bpl) + (bpl - 1); /* last byte, last row */
        prev = bits - bpl - 1;                  /* previous byte, previous row*/
        while (j > 0)
        {
            int i = bpl;

            while (i > 1)
            {
                if (*bits < *prev)
                    *bits = 0;
                else
                    *bits -= *prev;

                i--;
                bits--;
                prev--;
            }
            j--;
            bits--;
            prev--;
        }
    }
    else if (gmap->bitsPerPixel == 4)
    {
        int j = h;
        /* start at the last row and work up */
        bits = gmap->bits + (h * bpl) + (bpl - 1); /* last byte, last row */
        prev = bits - bpl;                      /* last byte, previous row*/
        while (j > 0)
        {
            int i = bpl;

            while (i > 0)
            {
                unsigned int p1, p2, n1, n2;

                p1 = *bits >> 4;    /* first nibble, last row */
                p2 = *bits & 0xF;   /* second nibble, last row */
                n2 = *prev-- >> 4;  /* second nibble, previous row */
                if (i == 1)
                    n1 = 0;         /* no nibble available */
                else
                    n1 = *prev & 0xF;/* first nibble, previous row */

                if (p1 < n1)
                    p1 = 0;
                else
                    p1 -= n1;

                if (p2 < n2)
                    p2 = 0;
                else
                    p2 -= n2;

                *bits-- = (FS_BYTE)((p1 << 4) | (p2 & 0xF));
                i--;
            }
            j--;
        }
    }
    else if (gmap->bitsPerPixel == 2)
    {
        int j = h;
        /* start at the last row and work up */
        bits = gmap->bits + (h * bpl) + (bpl - 1); /* last byte, last row */
        prev = bits - bpl;                      /* last byte, previous row*/
        while (j > 0)
        {
            int i = bpl;

            while (i > 0)
            {
                unsigned int p1, p2, p3, p4, n1, n2, n3, n4;

                p1 = *bits >> 6;         /* first pixel, last row  */
                p2 = (*bits >> 4) & 0x3; /* second pixel, last row */
                p3 = (*bits >> 2) & 0x3; /* third pixel, last row  */
                p4 = *bits & 0x3;        /* fourth pixel, last row */
                n4 = (*prev >> 2) & 0x3; /* fourth pixel, previous row   */
                n3 = (*prev >> 4) & 0x3; /* third pixel, previous row  */
                n2 = *prev-- >> 6;       /* second pixel, previous row   */
                if (i == 1)
                    n1 = 0;              /* no pixel available      */
                else
                    n1 = *prev & 0x3;    /* first pixel, previous row */

                if (p1 < n1)
                    p1 = 0;
                else
                    p1 -= n1;

                if (p2 < n2)
                    p2 = 0;
                else
                    p2 -= n2;

                if (p3 < n3)
                    p3 = 0;
                else
                    p3 -= n3;

                if (p4 < n4)
                    p4 = 0;
                else
                    p4 -= n4;

                *bits-- = (FS_BYTE)((p1 << 6) | (p2 << 4) | (p3 << 2) | (p4 & 0x3));
                i--;
            }
            j--;
        }
    }

    return gmap;
}


/****************************************************************/
/* emboss effect - light appears from upper right corner        */
/* subtract SE pixel and clip negative values                   */
FS_GRAYMAP *emboss_graymap(FS_GRAYMAP *gmap)
{
    FS_BYTE *bits, *next;
    FS_SHORT bpl;
    FS_SHORT h;

    if (gmap == 0)
        return 0;

    if ((gmap->height < 2) || (gmap->width < 2))
        return gmap;

    bpl = gmap->bpl;
    h = gmap->height - 1;

    if (gmap->bitsPerPixel == 8)
    {
        int j = 0;
        /* start at the first row and work down */
        bits = gmap->bits;       /* first pixel, first row */
        next = bits + bpl + 1;   /* next pixel, next row*/
        while (j < h)
        {
            int i = 1;

            while (i < bpl)
            {
                if (*bits < *next)
                    *bits = 0;
                else
                    *bits -= *next;

                i++;
                bits++;
                next++;
            }
            j++;
            bits++;
            next++;
        }
    }
    else if (gmap->bitsPerPixel == 4)
    {
        int j = 0;
        /* start at the first row and work down */
        bits = gmap->bits;  /* first byte, first row */
        next = bits + bpl;  /* same byte, next row*/
        while (j < h)
        {
            int i = 0;

            while (i < bpl)
            {
                unsigned int p1, p2, n1, n2;

                p1 = *bits >> 4;    /* first nibble, first row */
                p2 = *bits & 0xF;   /* second nibble, first row */
                n1 = *next++ & 0xF; /* first nibble, next row */
                if (i == bpl - 1)
                    n2 = 0;         /* no nibble available */
                else
                    n2 = *next >> 4;/* second nibble, next row */

                if (p1 < n1)
                    p1 = 0;
                else
                    p1 -= n1;

                if (p2 < n2)
                    p2 = 0;
                else
                    p2 -= n2;

                *bits++ = (FS_BYTE)((p1 << 4) | (p2 & 0xF));
                i++;
            }
            j++;
        }
    }
    else if (gmap->bitsPerPixel == 2)
    {
        int j = 0;
        /* start at the first row and work down */
        bits = gmap->bits;  /* first byte, first row */
        next = bits + bpl;  /* same byte, next row*/
        while (j < h)
        {
            int i = 0;

            while (i < bpl)
            {
                unsigned int p1, p2, p3, p4, n1, n2, n3, n4;

                p1 = *bits >> 6;         /* first pixel, first row  */
                p2 = (*bits >> 4) & 0x3; /* second pixel, first row */
                p3 = (*bits >> 2) & 0x3; /* third pixel, first row  */
                p4 = *bits & 0x3;        /* fourth pixel, first row */
                n1 = (*next >> 4) & 0x3; /* first pixel, next row   */
                n2 = (*next >> 2) & 0x3; /* second pixel, next row  */
                n3 = *next++ & 0x3;      /* third pixel, next row   */
                if (i == bpl - 1)
                    n4 = 0;              /* no pixel available      */
                else
                    n4 = *next >> 6;    /* fourth nibble, next row */

                if (p1 < n1)
                    p1 = 0;
                else
                    p1 -= n1;

                if (p2 < n2)
                    p2 = 0;
                else
                    p2 -= n2;

                if (p3 < n3)
                    p3 = 0;
                else
                    p3 -= n3;

                if (p4 < n4)
                    p4 = 0;
                else
                    p4 -= n4;

                *bits++ = (FS_BYTE)((p1 << 6) | (p2 << 4) | (p3 << 2) | (p4 & 0x3));
                i++;
            }
            j++;
        }
    }

    return gmap;
}

/*lint +e661 Warning -- Possible access of out-of-bounds pointer         */
/*lint +e662 Warning -- Possible creation of out-of-bounds pointer       */
/****************************************************************/
/* create an expanded byte image array from a graymap */
static FS_BYTE *GrowImage(_DS_ FS_GRAYMAP *src, FS_LONG rows, FS_LONG cols)
{
    FS_BYTE *dst;
    FS_BYTE *pdst, *psrc;
    FS_ULONG w, h;
    FS_LONG row, height;

    w = (FS_ULONG)(src->width + (cols << 1));
    h = (FS_ULONG)(src->height + (rows << 1));

    dst = FSS_calloc(_PS_ w * h);
    if (!dst)
        return 0;

    psrc = src->bits;
    pdst = dst + (rows * w) + cols;
    height = (FS_LONG)src->height;

    if (src->bitsPerPixel == 4)
    {
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0;
            FS_LONG w2 = src->width >> 1;
            while (i++ < w2)
            {
                *pdst++ = *psrc >> 4;
                *pdst++ = *psrc & 0xF;
                psrc++;
            }
            if (src->width & 1)
            {
                *pdst++ = *psrc >> 4;
                psrc++;
            }
            pdst += cols << 1;
        }
    }
    else if (src->bitsPerPixel == 8) /* direct copy */
    {
        for (row = 0; row < height; row++)
        {
            SYS_MEMCPY(pdst, psrc, src->width);
            psrc += src->bpl;
            pdst += w;
        }
    }
    else if (src->bitsPerPixel == 2)
    {
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0;
            FS_LONG w4 = src->width >> 2;
            FS_LONG m = src->width & 3;
            while (i++ < w4)
            {
                *pdst++ = (*psrc >> 6);
                *pdst++ = (*psrc >> 4) & 0x3;
                *pdst++ = (*psrc >> 2) & 0x3;
                *pdst++ = *psrc & 0x3;
                psrc++;
            }
            if (m == 1)
            {
                *pdst++ = (*psrc >> 6);
                psrc++;
            }
            else if (m == 2)
            {
                *pdst++ = (*psrc >> 6);
                *pdst++ = (*psrc >> 4) & 0x3;
                psrc++;
            }
            else if (m == 3)
            {
                *pdst++ = (*psrc >> 6);
                *pdst++ = (*psrc >> 4) & 0x3;
                *pdst++ = (*psrc >> 2) & 0x3;
                psrc++;
            }
            pdst += cols << 1;
        }
    }
    return dst;
}

/* remove the source image from an expanded image */
static FS_VOID RemoveSourceImage(FS_BYTE *dst, FS_GRAYMAP *src,
                                 FS_LONG rows, FS_LONG cols)
{
    FS_BYTE *pdst, *psrc;
    FS_ULONG w;
    FS_LONG height;

    w = (FS_ULONG)(src->width + (cols << 1));

    psrc = src->bits;
    pdst = dst + (rows * w) + cols;
    height = (FS_LONG)src->height;

    if (src->bitsPerPixel == 4)
    {
        FS_LONG row;
        FS_LONG w2 = src->width >> 1;
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0;
            while (i++ < w2)
            {
                *pdst++ -= *psrc >> 4;
                *pdst++ -= *psrc & 0xF;
                psrc++;
            }
            if (src->width & 1)
            {
                *pdst++ -= *psrc >> 4;
                psrc++;
            }
            pdst += cols << 1;
        }
    }
    else if (src->bitsPerPixel == 8) /* direct copy */
    {
        FS_LONG row;
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0;
            while (i++ < src->width)
            {
                *pdst++ -= *psrc++;
            }
            pdst += cols << 1;
        }
    }
    else if (src->bitsPerPixel == 2)
    {
        FS_LONG row;
        FS_LONG w4 = src->width >> 2;
        FS_LONG m = src->width & 3;
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0;
            while (i++ < w4)
            {
                *pdst++ -= (*psrc >> 6);
                *pdst++ -= (*psrc >> 4) & 0x3;
                *pdst++ -= (*psrc >> 2) & 0x3;
                *pdst++ -= *psrc & 0x3;
                psrc++;
            }
            if (m == 1)
            {
                *pdst++ -= (*psrc >> 6);
                psrc++;
            }
            else if (m == 2)
            {
                *pdst++ -= (*psrc >> 6);
                *pdst++ -= (*psrc >> 4) & 0x3;
                psrc++;
            }
            else if (m == 3)
            {
                *pdst++ -= (*psrc >> 6);
                *pdst++ -= (*psrc >> 4) & 0x3;
                *pdst++ -= (*psrc >> 2) & 0x3;
                psrc++;
            }
            pdst += cols << 1;
        }
    }
    return;
}

/* convert expanded byte image to graymap */
static FS_GRAYMAP *ImageToGraymap(_DS_ FS_BYTE *dst, FS_GRAYMAP *src,
                                  FS_ULONG width, FS_ULONG height, FS_USHORT npixel)
{
    FS_GRAYMAP *gmap;
    FS_BYTE *pbits, *pdst;
    FS_ULONG nsize, bpl;

    bpl = (width * src->bitsPerPixel + 7) / 8;

    nsize = offsetof(FS_GRAYMAP, bits);
    nsize += (FS_LONG)bpl * height;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    gmap = (FS_GRAYMAP *) FSS_calloc(_PS_ nsize);
    if (gmap == 0)
    {
        return 0;
    }

    /* set the fields */
    gmap->size = nsize;
    gmap->lo_x = src->lo_x - (npixel + 1);
    gmap->hi_y = src->hi_y + (npixel + 1);
    gmap->width = (FS_SHORT)width;
    gmap->bpl = (FS_SHORT)bpl;
    gmap->height = (FS_SHORT)height;
    gmap->i_dx = src->i_dx;
    gmap->i_dy = src->i_dy;
    gmap->dx = src->dx;
    gmap->dy = src->dy;
    gmap->bitsPerPixel = src->bitsPerPixel;
    gmap->type = src->type;

    /* set the bits */
    pbits = gmap->bits;
    pdst = dst;

    if (src->bitsPerPixel == 4)
    {
        FS_ULONG row;
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0, w2 = width >> 1;
            FS_ULONG hi, lo;
            while (i++ < w2)
            {
                hi = (*pdst++) << 4;
                lo = (*pdst++);
                *pbits++ = (FS_BYTE)(hi | lo);
            }
            if (width & 1)
            {
                hi = (FS_BYTE)(*pdst++) << 4;
                *pbits++ = (FS_BYTE)(hi);
            }
        }
    }
    else if (src->bitsPerPixel == 8) /* direct copy */
    {
        FS_ULONG row;
        for (row = 0; row < height; row++)
        {
            SYS_MEMCPY(pbits, pdst, width);
            pbits += width;
            pdst += width;
        }
    }
    else if (src->bitsPerPixel == 2)
    {
        FS_ULONG row;
        for (row = 0; row < height; row++)
        {
            FS_LONG i = 0, w4 = width >> 2, m = width & 3;
            FS_ULONG n1, n2, n3, n4;
            while (i++ < w4)
            {
                n1 = (*pdst++) << 6;
                n2 = (*pdst++) << 4;
                n3 = (*pdst++) << 2;
                n4 = (*pdst++);
                *pbits++ = (FS_BYTE)(n1 | n2 | n3 | n4);
            }
            if (m == 0)
                continue;
            else if (m == 1)
            {
                n1 = (*pdst++) << 6;
                *pbits++ = (FS_BYTE)(n1);
            }
            else if (m == 2)
            {
                n1 = (*pdst++) << 6;
                n2 = (*pdst++) << 4;
                *pbits++ = (FS_BYTE)(n1 | n2);
            }
            else if (m == 3)
            {
                n1 = (*pdst++) << 6;
                n2 = (*pdst++) << 4;
                n3 = (*pdst++) << 2;
                *pbits++ = (FS_BYTE)(n1 | n2 | n3);
            }
        }
    }

    return gmap;
}
/*lint +e661 Warning -- Possible access of out-of-bounds pointer         */
/*lint +e662 Warning -- Possible creation of out-of-bounds pointer       */

/****************************************************************/
/* n-pixel outline for graymaps                                 */
/****************************************************************/
FS_GRAYMAP *outline_graymap(_DS_ FS_GRAYMAP *src, FS_USHORT n, FS_FIXED opacity )
{
    FS_BYTE  *pr, *cr, *nr, *dp, dpmax;
    FS_LONG  bpl;
    FS_BYTE  *dst, *cpy;
    FS_LONG  npixel, width, height;
    FS_GRAYMAP *gmap;

    if (!src)
        return 0;

    if (src->width == 0 || src->height == 0)
        return src;

    dpmax = 0xF;
    if (src->bitsPerPixel == 8)
    {
        dpmax = 0xFF;
    }
    else if (src->bitsPerPixel == 2)
    {
        dpmax = 0x3;
    }

    npixel = n;

    /* extra top/bottom rows, left/right columns; so no "off the edge" testing needed */
    dst = GrowImage(_PS_ src, n + 1, n + 1);
    if (!dst)
        return 0;

    width  = src->width  + ((n + 1) << 1);
    height = src->height + ((n + 1) << 1);
    bpl = width;

    /* make copy */
    cpy = FSS_malloc(_PS_ width * height);
    if (!cpy)
    {
        FSS_free(_PS_ dst);
        return 0;
    }

    /* perform n-pixel dilation */
    while (npixel)
    {
        FS_LONG w = width - 1;
        FS_LONG h = height - 1;
        FS_LONG r, c;

        SYS_MEMCPY(cpy, dst, width * height);

#ifdef FS_SQUARE_OUTLINE
        if ( (npixel & 1))
#else
        if (!(npixel & 1))
#endif
        {
            /* odd iterations, an 8-neighbor dilate */
            pr = cpy;       /* previous row in cpy  */
            cr = pr + bpl;  /* current row in cpy   */
            nr = cr + bpl;  /* next row in cpy      */
            dp = dst + bpl;
            for (r = 1; r < h; r++)
            {
                FS_BYTE m1, m2, m3, m4, m5, m23;

                m4 = m23 = 0;
                m3 = MAX(pr[1], cr[1]); /* initialize queue */
                m2 = 0; /* due to padding added */
                for (c = 1; c < w; c++)
                {
                    if (c & 1)
                    {
                        /* use m2 and m3 from previous c */
                        m1 = m2;
                        m2 = m3;
                        m3 = MAX(pr[c + 1], cr[c + 1]);
                        m4 = MAX(nr[c + 1], nr[c]);
                        m5 = MAX(m4, nr[c - 1]);
                        m5 = MAX(m1, m5);
                        m23 = MAX(m2, m3);
                        dp[c] = (cr[c]) ? dpmax : MAX(m23, m5);
                    }
                    else
                    {
                        /* use m23, and m4 from previous c */
                        m2 = m3;
                        m3 = MAX(pr[c + 1], cr[c + 1]);
                        m5 = MAX(m4, nr[c + 1]);
                        m5 = MAX(m3, m5);
                        dp[c] = (cr[c]) ? dpmax : MAX(m23, m5);
                    }
                }
                pr += bpl;
                cr += bpl;
                nr += bpl;
                dp += bpl;
            }
        }
        else
        {
            /* even iterations, a 4-neighbor dilate */
            pr = cpy;       /* previous row in cpy  */
            cr = pr + bpl;  /* current row in cpy   */
            nr = cr + bpl;  /* next row in cpy      */
            dp = dst + bpl;
            for (r = 1; r < h; r++)
            {
                FS_BYTE mcr = 0, mpcn;
                for (c = 1; c < w; c++)
                {
                    if (c & 1)
                    {
                        mcr = MAX(cr[c], cr[c + 1]);
                        mpcn = MAX((MAX(pr[c], cr[c - 1])), nr[c]);
                        dp[c] = (cr[c]) ? dpmax : MAX(mcr, mpcn);
                    }
                    else
                    {
                        /* use mcr from previous c */
                        mpcn = MAX((MAX(pr[c], cr[c + 1])), nr[c]);
                        dp[c] = (cr[c]) ? dpmax : MAX(mcr, mpcn);
                    }
                }
                pr += bpl;
                cr += bpl;
                nr += bpl;
                dp += bpl;
            }
        }

        npixel--;
    }
    FSS_free(_PS_ cpy);

    /* remove interior if unfilled outline */
    if ((STATE.flags & FLAGS_OUTLINED_1PIXEL)  |
            (STATE.flags & FLAGS_OUTLINED_2PIXEL)  |
            (STATE.flags & FLAGS_OUTLINED_UNFILLED) )
    {
        RemoveSourceImage(dst, src, n + 1, n + 1);
    }

    /* adjust opacity if needed */
    if (opacity < FIXED_ONE)
    {
        FS_ULONG i, *lut;
        FS_ULONG size = width * height;
        FS_BYTE *pdst = dst;

        lut = FSS_malloc(_PS_ 256 * sizeof(FS_ULONG));
        if (lut)
        {
            for (i = 0; i < 256; i++)
            {
                lut[i] = (i * opacity) >> 16;
            }

            while (size--)
            {
                *pdst = (FS_BYTE)lut[*pdst];
                pdst++;
            }
            FSS_free(_PS_ lut);
        }
    }

    /* transfer dst back to src */
    gmap = ImageToGraymap(_PS_ dst, src, width, height, n);

    FSS_free(_PS_ dst);
    FSS_free_char(_PS_ src);

    return gmap;
}

/****************************************************************/
/* produce a softened N-pixel outline                           */
FS_GRAYMAP *soft_outline_graymap(_DS_ FS_GRAYMAP *gmap, FS_USHORT width,
                                 FS_FIXED opacity )
{
    gmap = outline_graymap(_PS_ gmap, width, FIXED_ONE);
    if (!gmap || STATE.error)
        return gmap; /* with error */

    gmap = Soften(_PS_ 0, 0, gmap);
    if (!gmap || STATE.error)
        return gmap; /* with error */

    /* apply opacity */
    if (STATE.outline_opacity != FIXED_ONE)
    {
        FS_ULONG i, *lut, lut16[16];
        FS_ULONG top = (1L << gmap->bitsPerPixel) - 1;
        FS_ULONG size = gmap->bpl * gmap->height;
        FS_BYTE *pdst = gmap->bits;

        if (top == 255)
            lut = FSS_malloc(_PS_ 256 * sizeof(FS_ULONG));
        else
            lut = lut16;

        if (lut)
        {

            for (i = 0; i < top + 1; i++)
            {
                lut[i] = (i * opacity) >> 16;
                if (lut[i] > top) lut[i] = top;
            }

            if (gmap->bitsPerPixel == 4)
            {
                FS_ULONG p1, p2;
                while (size--)
                {
                    p1 = (*pdst) >> 4;
                    p2 = (*pdst) & 0x0F;
                    p1 = lut[p1];  /* lut is capped at 0x0F */
                    p2 = lut[p2];
                    *pdst++ = (FS_BYTE)((p1 << 4) | p2);
                }
            }
            else if (gmap->bitsPerPixel == 8)
            {
                while (size--)
                {
                    *pdst = (FS_BYTE)lut[*pdst]; /* lut is capped at 0xFF */
                    pdst++;
                }
            }
            else
            {
                FS_ULONG p1, p2, p3, p4;
                while (size--)
                {
                    p1 =  (*pdst) >> 6;
                    p2 = ((*pdst) >> 4) & 0x03;
                    p3 = ((*pdst) >> 2) & 0x03;
                    p4 =  (*pdst) & 0x03;
                    p1 = lut[p1];  /* lut is capped at 0x03 */
                    p2 = lut[p2];
                    p3 = lut[p3];
                    p4 = lut[p4];
                    *pdst++ = (FS_BYTE)((p1 << 6) | (p2 << 4) | (p3 << 2) | p4);
                }
            }
            if (top == 255)FSS_free(_PS_ lut);
        }
    }
    return gmap;    /* possibly with error */
}

/****************************************************************/
/* filters to soften grayscale                                  */

/*
* Soft outline filter -- normally only used for the soft outline
* effect. This is a 5x5 symmetric filter that produces a very
* blurred output.
*/

static FS_CONST FS_SHORT f0[] = {5, 48, 48, 48, 48, 48};

/* 3x3 symmetric filters -- this is where you should start.
* the following filters differ in how much they spread out the
* gray in a single pixel among its neighbors.
*
* neutral filters (sum to 256) do not darken an image, but may
* 'wash out' single  pixel stems.  Darker filters boost the
* center value but also darken the overall character image.
*
* For stems thicker than 1 pixel, the neutral filter 3-9-3
* usually works just fine.  Clearly there are lots more filters
* you can try ... experiment to find out what looks best to you
* using your application on your hardware
*/

static FS_CONST FS_SHORT f1[] = {3, 25, 206, 25};    /* 1.5 12 1.5  (256) */
static FS_CONST FS_SHORT f2[] = {3, 34, 192, 34};    /* 2   11   2  (256) */
static FS_CONST FS_SHORT f3[] = {3, 51, 154, 51};    /* 3    9   3  (256) */
static FS_CONST FS_SHORT f4[] = {3, 65, 145, 65};    /* 4    9   4  (275) */

#ifdef FS_EXTRA_SOFTEN_FILTERS

static FS_CONST FS_SHORT f5[] = {3, 63, 156, 63};    /* 4   10   4  (282) */
static FS_CONST FS_SHORT f6[] = {3, 74, 148, 74};    /* 5   10   5  (296) */

static FS_CONST FS_SHORT f7[] = {3, 61, 167, 61};    /* 4   11   4  (289) */
static FS_CONST FS_SHORT f8[] = {3, 73, 159, 73};    /* 5   11   5  (305) */

static FS_CONST FS_SHORT f9[] = {3, 60, 178, 60};    /* 4   12   4  (298) */
static FS_CONST FS_SHORT f10[] = {3, 71, 169, 71};   /* 5   12   5  (311) */


/* if your application, hardware, or personal preference needs
* to spread the energy of a pixel out to more pixels than 3 wide
* filter -- try a 5 wide. These are typically used only on stems
* which are at least 3 pixels wide -- smaller stems turn to mush.
*
* Since the typical filter is <a,b,c,b,a> there are now three
* parameters to vary <a,b,c> which control the far, side, and center
* colors when applied to a 3 pixel wide black stem.
*
* There are LOTS and LOTS of 5 wide filters. Here are a few examples
*/

/* all neutral */
static FS_CONST FS_SHORT f11[] = {5, 16, 64, 96, 64, 16};     /* matches V2.2 smooth filter */
static FS_CONST FS_SHORT f12[] = {5, 18, 67, 86, 67, 18};     /* 1  4.5  9.5  15  9.5  4.5   1 */
static FS_CONST FS_SHORT f13[] = {5, 18, 59, 102, 59, 18};    /* 1    4  10   15  10     4   1 */
static FS_CONST FS_SHORT f14[] = {5, 18, 50, 120, 50, 18};    /* 1  3.5  10.5 15  10.5 3.5   1 */
static FS_CONST FS_SHORT f15[] = {5, 18, 42, 146, 42, 18};    /* 1    3  11   15  11     3   1 */
static FS_CONST FS_SHORT f16[] = {5, 18, 33, 154, 33, 18};    /* 1  2.5  11.5 15  11.5 2.5   1 */
static FS_CONST FS_SHORT f17[] = {5, 18, 25, 170, 25, 18};    /* 1    2  12   15  12     2   1 */
static FS_CONST FS_SHORT f18[] = {5, 18, 18, 184, 18, 18};    /* 1  1.5  12.5 15  12.5 1.5   1 */

/* OF COURSE, you can use different filters in X and Y ... making all sorts of
* 3x3, 5x5 and 3x5 or 5x3 filters.  Unfortunately we can't tell you what would
* look best on your system.  But here's a little guidance.  Start with a neutral
* 3x3 ... if it looks washed out go to a darkening filter.  If the 3 wide filter
* doesn't smooth enough to remove the artifacts in X or Y ... go to a neutral 5
* wide.  If it looks washed out ... go to a darkening 5 wide in that coordinate
*/

#endif /* FS_EXTRA_SOFTEN_FILTERS */

/* fx and fy are indices into the above array of <filters> */
FS_GRAYMAP *Soften(_DS_ int fx, int fy, FS_GRAYMAP *old )
{
    FS_SHORT width, height, bpl;
    FS_SHORT dx, dy;
    FS_ULONG size;
    FS_GRAYMAP * gmap;
    FS_SHORT FS_CONST *f;
    int r;
    FS_SHORT c;
    FS_SHORT bitsPerPixel = old->bitsPerPixel;
    FS_CONST FS_BYTE   mask_Bit[2] = {0xf0, 0x0f};
    FS_CONST FS_BYTE    set_Bit[2] = {0x4, 0x0};
    FS_CONST FS_BYTE  clear_Bit[2] = {0x0f, 0xf0};
    FS_CONST FS_BYTE  mask_Bit2[4] = {0xc0, 0x30, 0x0c, 0x03};
    FS_CONST FS_BYTE   set_Bit2[4] = {0x06, 0x04, 0x02, 0x00};
    FS_CONST FS_BYTE clear_Bit2[4] = {0x3F, 0xCF, 0xF3, 0xFC};

    /* move declaration inside Soften() to resolve portability issue */
    /* (pointers in static global data)                              */
#ifndef FS_EXTRA_SOFTEN_FILTERS
    FS_CONST FS_SHORT *filters[5];
#else
    FS_CONST FS_SHORT *filters[19];
#endif

    /* ? no filtering or bad filtering */
#ifndef FS_EXTRA_SOFTEN_FILTERS
    if ((fx < 0 && fy < 0) || fx > 4 || fy > 4)
        return old;
#else
    if ((fx < 0 && fy < 0) || fx > 18 || fy > 18)
        return old;
#endif

    filters[0] = f0;
    filters[1] = f1;
    filters[2] = f2;
    filters[3] = f3;
    filters[4] = f4;
#ifdef FS_EXTRA_SOFTEN_FILTERS
    filters[5] = f5;
    filters[6] = f6;
    filters[7] = f7;
    filters[8] = f8;
    filters[9] = f9;
    filters[10] = f10;
    filters[11] = f11;
    filters[12] = f12;
    filters[13] = f13;
    filters[14] = f14;
    filters[15] = f15;
    filters[16] = f16;
    filters[17] = f17;
    filters[18] = f18;
#endif

    /* change in size of result -- assumes filter width is ODD */
    dx = (filters[fx][0]) / 2;
    dy = (filters[fy][0]) / 2;

    /* size and allocate <gmap> */
    height = old->height + (dy << 1);
    width = old->width + (dx << 1);
    bpl = (width * bitsPerPixel + 7) / 8;
    size = offsetof(FS_GRAYMAP, bits);
    size += height * bpl;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    gmap = (FS_GRAYMAP *)FSS_calloc(_PS_ size);
    if (gmap == 0)
        return 0;

    gmap->size = size;
    gmap->width = width;
    gmap->height = height;
    gmap->bpl = bpl;
    gmap->i_dx = old->i_dx;
    gmap->i_dy = old->i_dy;
    gmap->dx = old->dx;
    gmap->dy = old->dy;
    gmap->hi_y = old->hi_y + dy;
    gmap->lo_x = old->lo_x - dx;
    gmap->bitsPerPixel = old->bitsPerPixel;
    gmap->type = old->type;

    /* horizontal pass */
    f = filters[fx];
    if (dx == 0)
    {
        FS_SHORT old_bpl = old->bpl;
        FS_SHORT gmap_bpl = gmap->bpl;
        FS_BYTE *old_bits = old->bits;
        FS_BYTE *gmap_bits = gmap->bits + gmap_bpl * dy;
        FS_SHORT h = old->height;
        for (r = 0; r < h; r++)
        {
            SYS_MEMCPY(gmap_bits, old_bits, old_bpl);
            old_bits += old_bpl;
            gmap_bits += gmap_bpl;
        }
    }
    else if (dx == 1)
    {
        FS_LONG v, a1, a2, a3;
        FS_SHORT gmap_bpl = gmap->bpl;
        FS_BYTE *pold = old->bits;
        FS_BYTE *pgmap = gmap->bits + gmap_bpl * dy;

        for (r = 0; r < old->height; r++)
        {
            a1 = 0;
            a2 = 0;
            a3 = 0;

            for (c = 0; c < old->width; c++)
            {
                if (bitsPerPixel == 4)
                {
                    a3 = (pold[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 15) v = 15;
                    pgmap[c >> 1] |= (v << set_Bit[c & 0x01]);
                }
                else if (bitsPerPixel == 8)
                {
                    a3 = pold[c];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 255) v = 255;
                    pgmap[c] = (FS_BYTE)v;
                }
                else if (bitsPerPixel == 2)
                {
                    a3 = (pold[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 3) v = 3;
                    pgmap[c >> 2] |= (v << set_Bit2[c & 0x03]);
                }
                a1 = a2;
                a2 = a3;
            }
            /* do last pixel of gmap in this row */
            v = (f[1] * a1 + f[2] * a2 + 128) >> 8;
            if (bitsPerPixel == 4)
            {
                if (v > 15) v = 15;
                pgmap[c >> 1] |= (v << set_Bit[c & 0x01]);
            }
            else if (bitsPerPixel == 8)
            {
                if (v > 255) v = 255;
                pgmap[c] = (FS_BYTE)v;
            }
            else if (bitsPerPixel == 2)
            {
                if (v > 3) v = 3;
                pgmap[c >> 2] |= (v << set_Bit2[c & 0x03]);
            }
            /* point to next row */
            pold += old->bpl;
            pgmap += gmap_bpl;
        }
    }
    else if (dx == 2)
    {
        FS_LONG v, a1, a2, a3, a4, a5;
        FS_SHORT old_bpl = old->bpl;
        FS_SHORT gmap_bpl = gmap->bpl;
        FS_BYTE *pold = old->bits;
        FS_BYTE *pgmap = gmap->bits + gmap_bpl * dy;

        for (r = 0; r < old->height; r++)
        {
            a1 = 0;
            a2 = 0;
            a3 = 0;
            a4 = 0;
            a5 = 0;

            for (c = 0; c < old->width; c++)
            {
                if (bitsPerPixel == 4)
                {
                    a5 = (pold[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 15) v = 15;
                    pgmap[c >> 1] |= (v << set_Bit[c & 0x01]);
                }
                else if (bitsPerPixel == 8)
                {
                    a5 = pold[c];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 255) v = 255;
                    pgmap[c] = (FS_BYTE)v;
                }
                else if (bitsPerPixel == 2)
                {
                    a5 = (pold[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 3) v = 3;
                    pgmap[c >> 2] |= (v << set_Bit2[c & 0x03]);
                }
                a1 = a2;
                a2 = a3;
                a3 = a4;
                a4 = a5;
            }
            /* do last two pixels of gmap in this row */
            v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
            if (bitsPerPixel == 4)
            {
                if (v > 15) v = 15;
                pgmap[c >> 1] |= (v << set_Bit[c & 0x01]);
                c++;
                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 15) v = 15;
                pgmap[c >> 1] |= (v << set_Bit[c & 0x01]);
            }
            else if (bitsPerPixel == 8)
            {
                if (v > 255) v = 255;
                pgmap[c] = (FS_BYTE)v;
                c++;
                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 255) v = 255;
                pgmap[c] = (FS_BYTE)v;
            }
            else if (bitsPerPixel == 2)
            {
                if (v > 3) v = 3;
                pgmap[c >> 2] |= (v << set_Bit2[c & 0x03]);
                c++;
                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 3) v = 3;
                pgmap[c >> 2] |= (v << set_Bit2[c & 0x03]);
            }

            /* point to next row */
            pold += old_bpl;
            pgmap += gmap_bpl;
        }
    }

    /* vertical pass */
    f = filters[fy];
    if (dy == 1)
    {
        FS_LONG v, a1, a2, a3;
        FS_SHORT gmap_bpl = gmap->bpl;
        FS_BYTE *psrc, *pdst;

        for ( c = 0; c < width; c++)
        {
            psrc = pdst = gmap->bits; /* point to first row */
            a1 = 1;
            a2 = 0;
            a3 = 0;
            /* value from row 0, column c */
            if (bitsPerPixel == 4)
                a2 = (psrc[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
            else if (bitsPerPixel == 8)
                a2 = psrc[c];
            else if (bitsPerPixel == 2)
                a2 = (psrc[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
            psrc += gmap_bpl; /* next row */

            for ( r = 0; r < (height - dy); r++ )
            {
                if (bitsPerPixel == 4)
                {
                    a3 = (psrc[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 15) v = 15;
                    pdst[c >> 1] &= clear_Bit[c & 0x01];
                    pdst[c >> 1] |= (v << set_Bit[c & 0x01]);
                }
                else if (bitsPerPixel == 8)
                {
                    a3 = psrc[c];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 255) v = 255;
                    pdst[c] = (FS_BYTE)v;
                }
                else if (bitsPerPixel == 2)
                {
                    a3 = (psrc[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + 128) >> 8;
                    if (v > 3) v = 3;
                    pdst[c >> 2] &= clear_Bit2[c & 0x03];
                    pdst[c >> 2] |= (v << set_Bit2[c & 0x03]);
                }
                a1 = a2;
                a2 = a3;
                psrc += gmap_bpl; /* advance to next row */
                pdst += gmap_bpl;
            }
            /* do the last row */
            v = (f[1] * a1 + f[2] * a2 + 128) >> 8;
            if (bitsPerPixel == 4)
            {
                if (v > 15) v = 15;
                pdst[c >> 1] &= clear_Bit[c & 0x01];
                pdst[c >> 1] |= (v << set_Bit[c & 0x01]);
            }
            else if (bitsPerPixel == 8)
            {
                if (v > 255) v = 255;
                pdst[c] = (FS_BYTE)v;
            }
            else if (bitsPerPixel == 2)
            {
                if (v > 3) v = 3;
                pdst[c >> 2] &= clear_Bit2[c & 0x03];
                pdst[c >> 2] |= (v << set_Bit2[c & 0x03]);
            }
        }
    }
    else if (dy == 2)
    {
        FS_LONG v, a1, a2, a3, a4, a5;
        FS_SHORT gmap_bpl = gmap->bpl;
        FS_BYTE *psrc, *pdst;

        for ( c = 0; c < width; c++)
        {
            psrc = pdst = gmap->bits; /* point to first row */
            a1 = 0;
            a2 = 0;

            if (bitsPerPixel == 4)
            {
                /* value from row 0, column c */
                a3 = (psrc[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                /* value from row 1, column c */
                psrc += gmap_bpl; /* next row */
                a4 = (psrc[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                psrc += gmap_bpl; /* next row */

                for ( r = 0; r < (height - dy); r++ )
                {
                    a5 = (psrc[c >> 1] & mask_Bit[c & 0x01]) >> set_Bit[c & 0x01];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 15) v = 15;
                    pdst[c >> 1] &= clear_Bit[c & 0x01];
                    pdst[c >> 1] |= (v << set_Bit[c & 0x01]);
                    a1 = a2;
                    a2 = a3;
                    a3 = a4;
                    a4 = a5;
                    psrc += gmap_bpl; /* advance to next row */
                    pdst += gmap_bpl;
                }
                /* do the last two rows */
                v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 15) v = 15;
                pdst[c >> 1] &= clear_Bit[c & 0x01];
                pdst[c >> 1] |= (v << set_Bit[c & 0x01]);

                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 15) v = 15;
                pdst += bpl; /* next old row */
                pdst[c >> 1] &= clear_Bit[c & 0x01];
                pdst[c >> 1] |= (v << set_Bit[c & 0x01]);
            }
            else if (bitsPerPixel == 8)
            {
                /* value from row 0, column c */
                a3 = psrc[c];
                /* value from row 1, column c */
                psrc += gmap_bpl; /* next row */
                a4 = psrc[c];
                psrc += gmap_bpl; /* next row */

                for ( r = 0; r < (height - dy); r++ )
                {
                    a5 = psrc[c];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 255) v = 255;
                    pdst[c] = (FS_BYTE)v;
                    a1 = a2;
                    a2 = a3;
                    a3 = a4;
                    a4 = a5;
                    psrc += gmap_bpl; /* advance to next row */
                    pdst += gmap_bpl;
                }
                /* do the last two rows */
                v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 255) v = 255;
                pdst[c] = (FS_BYTE)v;

                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 255) v = 255;
                pdst += bpl; /* next old row */
                pdst[c] = (FS_BYTE)v;
            }
            else if (bitsPerPixel == 2)
            {
                /* value from row 0, column c */
                a3 = (psrc[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                /* value from row 1, column c */
                psrc += gmap_bpl; /* next row */
                a4 = (psrc[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                psrc += gmap_bpl; /* next row */

                for ( r = 0; r < (height - dy); r++ )
                {
                    a5 = (psrc[c >> 2] & mask_Bit2[c & 0x03]) >> set_Bit2[c & 0x03];
                    v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + f[5] * a5 + 128) >> 8;
                    if (v > 3) v = 3;
                    pdst[c >> 2] &= clear_Bit2[c & 0x03];
                    pdst[c >> 2] |= (v << set_Bit2[c & 0x03]);
                    a1 = a2;
                    a2 = a3;
                    a3 = a4;
                    a4 = a5;
                    psrc += gmap_bpl; /* advance to next row */
                    pdst += gmap_bpl;
                }
                /* do the last two rows */
                v = (f[1] * a1 + f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 3) v = 3;
                pdst[c >> 2] &= clear_Bit2[c & 0x03];
                pdst[c >> 2] |= (v << set_Bit2[c & 0x03]);

                v = (f[2] * a2 + f[3] * a3 + f[4] * a4 + 128) >> 8;
                if (v > 3) v = 3;
                pdst += bpl; /* next old row */
                pdst[c >> 2] &= clear_Bit2[c & 0x03];
                pdst[c >> 2] |= (v << set_Bit2[c & 0x03]);
            }
        }
    }

    FSS_free_char( _PS_ old );
    return gmap;
}

#endif /* FS_GRAYMAPS */

#if defined(FS_GRAYMAPS) || defined(FS_EDGE_RENDER)
#ifdef FS_PSEUDO_BOLD
/*lint -e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */

#define pixelvalue(ptr,i) ((ptr[(i)>>indexshift]&mask[(i)&indexmask])>>(rightshift[(i)&indexmask]))
#define setpixel(i,value) ((ptr[(i)>>indexshift]) = (ptr[(i)>>indexshift]&unmask[(i)&indexmask])|((value)<<(leftshift[(i)&indexmask])))

FS_GRAYMAP *pixelbold_graymap2(_DS_ FS_GRAYMAP *gmap)
{
    int row, col, bpl;
    unsigned char   mask[4] = {0xc0, 0x30, 0x0c, 0x03};
    unsigned char unmask[4] = {0x3f, 0xcf, 0xf3, 0xfc};

    unsigned int rightshift[4] = {6, 4, 2, 0};
    unsigned int  leftshift[4] = {6, 4, 2, 0};
    int indexshift = 2;
    int indexmask = 0x3;
    int blackpixel = 0x3;
    int quarterpixel = 0x2;
    FS_BYTE *ptr;
    FS_BYTE *rowabove;
    FS_BYTE *rowbelow;
    FS_LONG bold_width = STATE.cur_sfnt->senv->bold_width;

    if (bold_width == 0)
        return gmap;

    bpl = gmap->bpl;
    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        pixel1 = 0;
        pixel2 = pixelvalue(ptr, 0);
        pixel3 = pixelvalue(ptr, 1);
        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            col = 1;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col + 1; i < gmap->width; i++)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j < i; j++)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j <= indexofdarkest; j++)
                    {
                        int val = pixelvalue(ptr, j);
                        setpixel(j - 1, (FS_BYTE)val);
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }

            }
        }
        for (col = 2; col < gmap->width; col++)
        {
            int i;

            pixel1 = pixelvalue(ptr, col - 2);
            pixel2 = pixelvalue(ptr, col - 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col + 1; i < gmap->width; i++)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                    abovepixel3 == blackpixel &&
                                    belowpixel1 == 0 &&
                                    belowpixel2 == 0 &&
                                    belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col - 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j < i; j++)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }
                    if (darkest > quarterpixel)
                    {
                        for (j = col; j <= indexofdarkest; j++)
                        {
                            setpixel(j - 1, pixelvalue(ptr, j));
                        }

                        /* make darkest pixel black */
                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
        }
    }

    if (bold_width < 2 )
        return gmap;


    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        col = gmap->width - 2;
        pixel1 = 0;
        pixel2 = pixelvalue(ptr, col + 1);
        pixel3 = pixelvalue(ptr, col);

        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col - 1; i >= 0; i--)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j > i; j--)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j >= indexofdarkest; j--)
                    {
                        setpixel(j + 1, pixelvalue(ptr, j));
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }
            }
        }
        for (col = gmap->width - 3; col >= 0; col--)
        {
            int i;

            pixel1 = pixelvalue(ptr, col + 2);
            pixel2 = pixelvalue(ptr, col + 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col - 1; i >= 0; i--)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                abovepixel3 == blackpixel &&
                                belowpixel1 == 0 &&
                                belowpixel2 == 0 &&
                                belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col + 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j > i; j--)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }

                    if (darkest > quarterpixel)
                    {
                        for (j = col; j >= indexofdarkest; j--)
                        {
                            setpixel(j + 1, pixelvalue(ptr, j));
                        }

                        /* make darkest pixel black */
                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
            else if (row != 0 && row != gmap->height - 1)
            {
                /* special case of horizontal stroke meeting a vertical stroke. Check
                ** row above and below to determine if pixel should be set
                */
                if (pixel1 == blackpixel && pixel2 == 0 && pixel3 == blackpixel)
                {
                    int abovepixel1 = pixelvalue(rowabove, col - 2);
                    /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                    int abovepixel3 = pixelvalue(rowabove, col);
                    int belowpixel1 = pixelvalue(rowbelow, col - 2);
                    int belowpixel2 = pixelvalue(rowbelow, col - 1);
                    int belowpixel3 = pixelvalue(rowbelow, col);

                    if (abovepixel1 == 0 &&
                        abovepixel3 == blackpixel &&
                        belowpixel1 == 0 &&
                        belowpixel2 == 0 &&
                        belowpixel3 == blackpixel)
                        setpixel(col - 1, (FS_BYTE)blackpixel);
                }
            }
        }
    }
    return gmap;
}

FS_GRAYMAP *pixelbold_graymap4(_DS_ FS_GRAYMAP *gmap)
{
    int row, col, bpl;
    unsigned char mask[2] = {0xf0, 0x0f};
    unsigned char unmask[2] = {0x0f, 0xf0};
    unsigned int rightshift[2] = {4, 0};
    unsigned int leftshift[2] = {4, 0};
    int indexshift = 1;
    int indexmask = 0x1;
    int blackpixel = 0xf;
    int quarterpixel = 0x4;
    FS_BYTE *ptr;
    FS_BYTE *rowabove;
    FS_BYTE *rowbelow;
    FS_LONG bold_width = STATE.cur_sfnt->senv->bold_width;

    if (bold_width == 0)
        return gmap;

    bpl = gmap->bpl;
    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        pixel1 = 0;
        pixel2 = pixelvalue(ptr, 0);
        pixel3 = pixelvalue(ptr, 1);
        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            col = 1;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col + 1; i < gmap->width; i++)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j < i; j++)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j <= indexofdarkest; j++)
                    {
                        int val = pixelvalue(ptr, j);
                        setpixel(j - 1, (FS_BYTE)val);
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }
            }
        }
        for (col = 2; col < gmap->width; col++)
        {
            int i;

            pixel1 = pixelvalue(ptr, col - 2);
            pixel2 = pixelvalue(ptr, col - 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col + 1; i < gmap->width; i++)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                abovepixel3 == blackpixel &&
                                belowpixel1 == 0 &&
                                belowpixel2 == 0 &&
                                belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col - 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j < i; j++)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }
                    if (darkest > quarterpixel)
                    {
                        for (j = col; j <= indexofdarkest; j++)
                        {
                            int val = pixelvalue(ptr, j);
                            setpixel(j - 1, (FS_BYTE)val);
                        }

                        /* make darkest pixel black */
                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
        }
    }

    if (bold_width < 2 )
        return gmap;

    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        col = gmap->width - 2;
        pixel1 = 0;
        pixel2 = pixelvalue(ptr, col + 1);
        pixel3 = pixelvalue(ptr, col);

        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col - 1; i >= 0; i--)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j > i; j--)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j >= indexofdarkest; j--)
                    {
                        int val = pixelvalue(ptr, j);
                        setpixel(j + 1, (FS_BYTE)val);
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }
            }
        }
        for (col = gmap->width - 3; col >= 0; col--)
        {
            int i;

            pixel1 = pixelvalue(ptr, col + 2);
            pixel2 = pixelvalue(ptr, col + 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col - 1; i >= 0; i--)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                abovepixel3 == blackpixel &&
                                belowpixel1 == 0 &&
                                belowpixel2 == 0 &&
                                belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col + 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j > i; j--)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }

                    if (darkest > quarterpixel)
                    {
                        for (j = col; j >= indexofdarkest; j--)
                        {
                            int val = pixelvalue(ptr, j);
                            setpixel(j + 1, (FS_BYTE)val);
                        }

                        /* make darkest pixel black */
                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
            else if (row != 0 && row != gmap->height - 1)
            {
                /* special case of horizontal stroke meeting a vertical stroke. Check
                ** row above and below to determine if pixel should be set
                */
                if (pixel1 == blackpixel && pixel2 == 0 && pixel3 == blackpixel)
                {
                    int abovepixel1 = pixelvalue(rowabove, col - 2);
                    /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                    int abovepixel3 = pixelvalue(rowabove, col);
                    int belowpixel1 = pixelvalue(rowbelow, col - 2);
                    int belowpixel2 = pixelvalue(rowbelow, col - 1);
                    int belowpixel3 = pixelvalue(rowbelow, col);

                    if (abovepixel1 == 0 &&
                        abovepixel3 == blackpixel &&
                        belowpixel1 == 0 &&
                        belowpixel2 == 0 &&
                        belowpixel3 == blackpixel)
                        setpixel(col - 1, (FS_BYTE)blackpixel);
                }
            }
        }
    }
    return gmap;
}

#undef  pixelvalue
#undef  setpixel
#define pixelvalue(ptr,i) ((ptr[(i)]&0xff))
#define setpixel(i,value) (ptr[(i)] = (value))

FS_GRAYMAP *pixelbold_graymap8(_DS_ FS_GRAYMAP *gmap)
{
    int row, col, bpl;
    int blackpixel = 0xff;
    int quarterpixel = 0x10;
    FS_BYTE *ptr;
    FS_BYTE *rowabove;
    FS_BYTE *rowbelow;
    FS_LONG bold_width = STATE.cur_sfnt->senv->bold_width;

    if (bold_width == 0)
        return gmap;

    bpl = gmap->bpl;
    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        pixel1 = 0;
        pixel2 = pixelvalue(ptr, 0);
        pixel3 = pixelvalue(ptr, 1);
        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            col = 1;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col + 1; i < gmap->width; i++)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j < i; j++)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j <= indexofdarkest; j++)
                    {
                        int val = pixelvalue(ptr, j);
                        setpixel(j - 1, (FS_BYTE)val);
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }
            }
        }

        for (col = 2; col < gmap->width; col++)
        {
            int i;

            pixel1 = pixelvalue(ptr, col - 2);
            pixel2 = pixelvalue(ptr, col - 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col + 1; i < gmap->width; i++)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                abovepixel3 == blackpixel &&
                                belowpixel1 == 0 &&
                                belowpixel2 == 0 &&
                                belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col - 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j < i; j++)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }
                    if (darkest > quarterpixel)
                    {
                        for (j = col; j <= indexofdarkest; j++)
                        {
                            int val = pixelvalue(ptr, j);
                            setpixel(j - 1, (FS_BYTE)val);
                        }
                        /* make darkest pixel black */

                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
        }
    }

    if (bold_width < 2 )
        return gmap;


    for (row = 0; row < gmap->height; row++)
    {
        int pixel1;
        int pixel2;
        int pixel3;
        int nextpixel;

        ptr = gmap->bits + row * bpl;
        rowabove = gmap->bits + (row - 1) * bpl;
        rowbelow = gmap->bits + (row + 1) * bpl;

        col = gmap->width - 2;
        pixel1 = 0;
        pixel2 = pixelvalue(ptr, col + 1);
        pixel3 = pixelvalue(ptr, col);

        /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
        if (pixel2 == 0 && pixel3 != 0)
        {
            int i;
            /* found the left edge of a stroke, now find the right edge */
            for (i = col - 1; i >= 0; i--)
            {
                nextpixel = pixelvalue(ptr, i);
                if (nextpixel == 0)
                    break;
            }

            if (pixel1 == blackpixel)
            {
                /* if edge is black, simply add an additional black pixel */
                setpixel(col - 1, (FS_BYTE)blackpixel);
                /* set index for next stroke and continue */
            }
            else
            {
                /* if edge pixel is gray, find darkest pixel and duplicate darkest
                pixel and shift the pixels left */
                int darkest = pixel3;
                int indexofdarkest = col;
                int j;
                for (j = col; j > i; j--)
                {
                    if (pixelvalue(ptr, j) > darkest)
                    {
                        darkest = pixelvalue(ptr, j);
                        indexofdarkest = j;
                    }
                }

                if (darkest > quarterpixel)
                {
                    for (j = col; j >= indexofdarkest; j--)
                    {
                        int val = pixelvalue(ptr, j);
                        setpixel(j + 1, (FS_BYTE)val);
                    }

                    /* make darkest pixel black */
                    setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                }
            }
        }

        for (col = gmap->width - 3; col >= 0; col--)
        {
            int i;

            pixel1 = pixelvalue(ptr, col + 2);
            pixel2 = pixelvalue(ptr, col + 1);
            pixel3 = pixelvalue(ptr, col);
            /* locate the next stroke where a stroke is a set of non-white pixels between two white pixels.*/
            if (pixel2 == 0 && pixel3 != 0)
            {
                /* found the left edge of a stroke, now find the right edge */
                for (i = col - 1; i >= 0; i--)
                {
                    nextpixel = pixelvalue(ptr, i);
                    if (nextpixel == 0)
                        break;
                }

                /* if there's no room to embolden, set index for next stroke and skip */
                if (pixel1 != 0)
                {
                    if (row != 0 && row != gmap->height - 1)
                    {
                        /* special case of horizontal stroke meeting a vertical stroke. Check
                        ** row above and below to determine if pixel should be set
                        */
                        if (pixel1 >= 12 && pixel2 == 0 && pixel3 == blackpixel)
                        {
                            int abovepixel1 = pixelvalue(rowabove, col - 2);
                            /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                            int abovepixel3 = pixelvalue(rowabove, col);
                            int belowpixel1 = pixelvalue(rowbelow, col - 2);
                            int belowpixel2 = pixelvalue(rowbelow, col - 1);
                            int belowpixel3 = pixelvalue(rowbelow, col);

                            if (abovepixel1 == 0 &&
                                abovepixel3 == blackpixel &&
                                belowpixel1 == 0 &&
                                belowpixel2 == 0 &&
                                belowpixel3 == blackpixel)
                                setpixel(col - 1, (FS_BYTE)blackpixel);
                        }
                    }

                    col = i;
                    continue;
                }

                if (pixel1 == blackpixel)
                {
                    /* if edge is black, simply add an additional black pixel */
                    setpixel(col + 1, (FS_BYTE)blackpixel);
                    /* set index for next stroke and continue */
                }
                else
                {
                    /* if edge pixel is gray, find darkest pixel and duplicate darkest
                    pixel and shift the pixels left */
                    int darkest = pixel3;
                    int indexofdarkest = col;
                    int j;
                    for (j = col; j > i; j--)
                    {
                        if (pixelvalue(ptr, j) > darkest)
                        {
                            darkest = pixelvalue(ptr, j);
                            indexofdarkest = j;
                        }
                    }

                    if (darkest > quarterpixel)
                    {
                        for (j = col; j >= indexofdarkest; j--)
                        {
                            int val = pixelvalue(ptr, j);
                            setpixel(j + 1, (FS_BYTE)val);
                        }

                        /* make darkest pixel black */
                        setpixel(indexofdarkest, (FS_BYTE)blackpixel);
                    }
                }
                col = i;
            }
            else if (row != 0 && row != gmap->height - 1)
            {
                /* special case of horizontal stroke meeting a vertical stroke. Check
                ** row above and below to determine if pixel should be set
                */
                if (pixel1 == blackpixel && pixel2 == 0 && pixel3 == blackpixel)
                {
                    int abovepixel1 = pixelvalue(rowabove, col - 2);
                    /*int abovepixel2 = pixelvalue(rowabove,col-1);*/
                    int abovepixel3 = pixelvalue(rowabove, col);
                    int belowpixel1 = pixelvalue(rowbelow, col - 2);
                    int belowpixel2 = pixelvalue(rowbelow, col - 1);
                    int belowpixel3 = pixelvalue(rowbelow, col);

                    if (abovepixel1 == 0 &&
                        abovepixel3 == blackpixel &&
                        belowpixel1 == 0 &&
                        belowpixel2 == 0 &&
                        belowpixel3 == blackpixel)
                        setpixel(col - 1, (FS_BYTE)blackpixel);
                }
            }
        }
    }
    return gmap;
}


FS_GRAYMAP *pixelbold_embedded_graymap(_DS_ FS_GRAYMAP *gmap)
{
    SFNT *sfnt = STATE.cur_sfnt;
    FS_SHORT bw = sfnt->senv->bold_width;
    FS_GRAYMAP *new_gmap, *cp_new;
    FS_SHORT width;
    FS_SHORT height, row, col;
    FS_LONG size;
    FS_BYTE *src_bits, *dst_bits;
    FS_SHORT bpl = 0;
    int bit_added = 0;
    int bplshift = 0;

    if (bw == 0)
        return gmap;

    if (bw == 1)
        bit_added = 1;
    else
        bit_added = 2;

    width = gmap->width + (FS_SHORT)bit_added;
    height = gmap->height;

    if (gmap->bitsPerPixel == 4)
    {
        bpl = (1 + width) >> 1;
        bplshift = 4;
    }
    else if (gmap->bitsPerPixel == 8)
    {
        bpl = width;
        bplshift = 8;
    }
    else if (gmap->bitsPerPixel == 2)
    {
        bpl = (3 + width) >> 2;
        bplshift = 2;
    }

    size = offsetof(FS_GRAYMAP, bits);
    size += height * bpl;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    new_gmap = (FS_GRAYMAP *) FSS_calloc(_PS_ size);
    if (new_gmap == 0)
    {
        FSS_free_char(_PS_ gmap);
        return 0;
    }

    new_gmap->size = size;
    new_gmap->width = width;
    new_gmap->height = height;
    new_gmap->bpl = bpl;
    new_gmap->i_dx = gmap->i_dx;
    new_gmap->i_dy = gmap->i_dy;
    new_gmap->dx = gmap->dx;
    new_gmap->dy = gmap->dy;
    new_gmap->hi_y = gmap->hi_y;
    new_gmap->lo_x = gmap->lo_x - 1;
    new_gmap->bitsPerPixel = gmap->bitsPerPixel;

    src_bits = gmap->bits;
    dst_bits = new_gmap->bits;

    for (row = 0; row < gmap->height; row++)
    {
        SYS_MEMCPY(dst_bits, src_bits, gmap->bpl);
        src_bits += gmap->bpl;
        dst_bits += new_gmap->bpl;
    }

    cp_new = copy_graymap(_PS_ new_gmap);
    if (cp_new == 0)
    {
        FSS_free(_PS_ new_gmap);
        return 0;
    }

    src_bits = cp_new->bits;
    dst_bits = new_gmap->bits;
    for (row = 0; row < new_gmap->height; row++)
    {
        src_bits = cp_new->bits + row * bpl;
        dst_bits = new_gmap->bits + row * bpl;
        dst_bits[0] = src_bits[0] >> bplshift;
        for (col = 1; col < bpl; col++)
            dst_bits[col] = (src_bits[col - 1] << (8 - bplshift)) | (src_bits[col] >> bplshift); /*lint !e661*/
    }

    FSS_free_char(_PS_ gmap);
    FSS_free_char(_PS_ cp_new);

    if (new_gmap->bitsPerPixel == 4)
        pixelbold_graymap4(_PS_ new_gmap);
    else if (new_gmap->bitsPerPixel == 8)
        pixelbold_graymap8(_PS_ new_gmap);
    else if (new_gmap->bitsPerPixel == 2)
        pixelbold_graymap2(_PS_ new_gmap);

    return new_gmap;
}

/*lint +e850  Warning 850: Warning -- for loop index variable is modified in body of the for loop */

#endif /* FS_PSEUDO_BOLD */

#endif /* FS_GRAYMAPS or FS_EDGE_RENDER */
